using Server.Items;
using Server.Mobiles;
using Server.Network;
using System;
using System.Collections;

namespace Server.Engines.XmlSpawner2
{
    public class XmlSpecial : XmlAttachment
    {
        public enum AbilityType
        {
            Random,
            Decimate,
            Banish,
            PoisonPool,
            LivingBomb,
            FlareVortex,
            Plague,
            Molten,
            ColdBlood
        }

        private AbilityType m_AbilityType;
        private TimeSpan m_Refractory = TimeSpan.FromSeconds(40.0);
        private DateTime m_EndTime;
        private int m_Hue = 193; //barva textu ability

        [CommandProperty(AccessLevel.GameMaster)]
        public AbilityType Ability { get { return m_AbilityType; } set { m_AbilityType = value; } }

        // These are the various ways in which the message attachment can be constructed.
        // These can be called via the [addatt interface, via scripts, via the spawner ATTACH keyword.
        // Other overloads could be defined to handle other types of arguments

        // a serial constructor is REQUIRED
        public XmlSpecial(ASerial serial) : base(serial)
        {
        }

        [Attachable]
        public XmlSpecial()
        {
            m_AbilityType = AbilityType.Random;
        }

        [Attachable]
        public XmlSpecial(AbilityType ability)
        {
            m_AbilityType = ability;
        }

        // note that this method will be called when attached to either a mobile or a weapon
        // when attached to a weapon, only that weapon will do additional damage
        // when attached to a mobile, any weapon the mobile wields will do additional damage
        public override void OnDamage(Mobile mob, int amount, Mobile from, bool willKill, int typ)//OnWeaponHit(Mobile attacker, Mobile defender, BaseWeapon weapon, int damageGiven)
        {
            // if it is still refractory then return
            if (DateTime.Now < m_EndTime || from == null || mob == null || willKill || typ == 1) return;

            AbilityType abi = m_AbilityType;

            if (mob.Hits > ((mob.HitsMax * 80) / 100)) return;

            if (m_AbilityType == AbilityType.Random)
            {
                abi = RandomAbilityType();
            }

            m_Refractory = TimeSpan.FromSeconds(Utility.RandomMinMax(30, 60));

            if (abi == AbilityType.Decimate)
            {
                mob.PublicOverheadMessage(MessageType.Regular, m_Hue, true, "Decimate!");
                AbilitySound(mob);
                Timer.DelayCall(TimeSpan.FromSeconds(2.0), new TimerStateCallback(TriggerDecimate), new object[] { mob });
                m_EndTime = DateTime.Now + m_Refractory;
            }
            else if (abi == AbilityType.Banish)
            {
                mob.PublicOverheadMessage(MessageType.Regular, m_Hue, true, "Banish!");
                AbilitySound(mob);
                mob.CantWalk = true;
                Timer.DelayCall(TimeSpan.FromSeconds(5.0), new TimerStateCallback(TriggerBanish), new object[] { mob });
                m_EndTime = DateTime.Now + m_Refractory;
            }
            else if (abi == AbilityType.PoisonPool)
            {
                mob.PublicOverheadMessage(MessageType.Regular, m_Hue, true, "Pool of Poison!");
                mob.CantWalk = true;
                AbilitySound(mob);
                Timer.DelayCall(TimeSpan.FromSeconds(5.0), new TimerStateCallback(TriggerPoisonPool), new object[] { mob });
                m_EndTime = DateTime.Now + TimeSpan.FromSeconds(30.0);
            }
            else if (abi == AbilityType.LivingBomb)
            {
                TriggerLivingBomb(mob);
                m_EndTime = DateTime.Now + m_Refractory;
            }
            else if (abi == AbilityType.FlareVortex)
            {
                bool validLocation = false;
                Point3D loc = mob.Location;

                for (int j = 0; !validLocation && j < 10; ++j)
                {
                    int x = mob.X + Utility.Random(6);
                    int y = mob.Y + Utility.Random(6);
                    int z = mob.Map.GetAverageZ(x, y);

                    if (validLocation = (mob.Map.CanFit(x, y, mob.Z, 16, false, false) && mob.InLOS(new Point3D(x, y, mob.Z))))
                        loc = new Point3D(x, y, mob.Z);
                    else if (validLocation = (mob.Map.CanFit(x, y, z, 16, false, false) && mob.InLOS(new Point3D(x, y, z))))
                        loc = new Point3D(x, y, z);
                }

                if (!validLocation)
                    return;

                mob.PublicOverheadMessage(MessageType.Regular, m_Hue, true, "Flare Vortex!");
                AbilitySound(mob);
                Effects.SendMovingEffect(mob, (IEntity)new Entity(Server.Serial.Zero, loc, mob.Map), 14228, 7, 0, false, false, 2021, 0);

                FlareVortex flare = (FlareVortex)new FlareVortex();
                Timer.DelayCall(TimeSpan.FromSeconds(1.0), new TimerStateCallback(SpawnFlare), new object[] { flare, loc, mob.Map });
                m_EndTime = DateTime.Now + m_Refractory;
            }
            else if (abi == AbilityType.Plague)
            {
                ArrayList targets = Utilitky.GetTargets(mob, 15, false);
                mob.PublicOverheadMessage(MessageType.Regular, m_Hue, true, "Wandering Plague!");

                for (int i = 0; i < targets.Count; ++i)
                {
                    Mobile m = (Mobile)targets[i];

                    if (m == null || m.Deleted || KhaldunSummoner.IsUnderInfluence(m))
                        continue;

                    KhaldunSummoner.Plague(m, mob, false);
                    m_EndTime = DateTime.Now + m_Refractory;
                }
            }
            else if (abi == AbilityType.Molten)
            {
                mob.PublicOverheadMessage(MessageType.Regular, m_Hue, true, "Molten Ground!");
                AbilitySound(mob);
                mob.CantWalk = true;
                Timer.DelayCall(TimeSpan.FromSeconds(3.0), new TimerStateCallback(TriggerMolten), new object[] { mob });
                m_EndTime = DateTime.Now + TimeSpan.FromSeconds(30.0);
            }
            else if (abi == AbilityType.ColdBlood)
            {
                AbilitySound(mob);
                mob.CantWalk = true;
                Timer.DelayCall(TimeSpan.FromSeconds(2.0), new TimerStateCallback(TriggerColdBlood), new object[] { mob });
                m_EndTime = DateTime.Now + m_Refractory;
            }
            else
                return;
        }

        public override void Serialize(GenericWriter writer)
        {
            base.Serialize(writer);

            writer.Write((int)0);

            writer.Write((int)m_AbilityType);
            writer.Write(m_EndTime - DateTime.Now);

        }

        public override void Deserialize(GenericReader reader)
        {
            base.Deserialize(reader);

            int version = reader.ReadInt();
            switch (version)
            {
                // 				case 1:
                // 					Range = reader.ReadInt();
                // 					goto case 0;
                case 0:
                    // version 0
                    m_AbilityType = (AbilityType)reader.ReadInt();
                    TimeSpan remaining = reader.ReadTimeSpan();
                    m_EndTime = DateTime.Now + remaining;
                    break;
            }
        }

        public override string OnIdentify(Mobile from)
        {
            return "Special Ability";
        }

        private static AbilityType RandomAbilityType()
        {
            AbilityType rnd = AbilityType.Decimate;

            switch (Utility.Random(8))
            {
                case 0: rnd = AbilityType.Decimate; break;
                case 1: rnd = AbilityType.Banish; break;
                case 2: rnd = AbilityType.PoisonPool; break;
                case 3: rnd = AbilityType.LivingBomb; break;
                case 4: rnd = AbilityType.FlareVortex; break;
                case 5: rnd = AbilityType.Plague; break;
                case 6: rnd = AbilityType.Molten; break;
                case 7: rnd = AbilityType.ColdBlood; break;
            }

            return rnd;
        }

        public static int PercentualDamage(int percent, Mobile mob)
        {
            int damage = 1;

            if (mob != null && percent > 1)
            {
                damage = ((mob.HitsMax * percent) / 100);
            }

            return damage;
        }

        private static void AbilitySound(Mobile mob)
        {
            if (mob == null) return;

            if (mob.Body.IsHuman)
            {
                if (mob.Female)
                    mob.PlaySound(Utility.RandomList(778, 779, 794, 796, 803, 811, 812));
                else
                    mob.PlaySound(Utility.RandomList(1049, 1050, 1066, 1068, 1073, 1075, 1085, 1086));
            }
            else
                mob.PlaySound(957);
        }

        #region DECIMATE
        private void TriggerDecimate(object state)
        {
            object[] args = (object[])state;
            Mobile mob = (Mobile)args[0];

            if (mob == null || mob.Deleted || !mob.Alive)
                return;

            ArrayList targets = Utilitky.GetTargets(mob, 12, false);

            for (int i = 0; i < targets.Count; ++i)
            {
                Mobile m = (Mobile)targets[i];

                if (m == null || m.Deleted)
                    continue;

                mob.DoHarmful(m);
                DecimateTarget(new object[] { 0, m, mob, (int)PercentualDamage(25, m) });
            }
        }

        private void DecimateTarget(object state)
        {
            object[] args = (object[])state;
            int count = (int)args[0];
            Mobile mob = (Mobile)args[1];
            Mobile attachedMob = (Mobile)args[2];
            int dmg = (int)args[3];
            bool decimateHim = false;

            if (attachedMob == null || attachedMob.Deleted || !attachedMob.Alive || mob == null || mob.Deleted || !mob.Alive || !attachedMob.InRange(mob.Location, 20))
                return;

            if (mob is BaseCreature && (((BaseCreature)mob).Controlled || ((BaseCreature)mob).Summoned))
            {
                if (!mob.IsDeadBondedPet && mob.Alive && !mob.Blessed && mob.InLOS(attachedMob))
                    decimateHim = true;
            }
            else if (mob.Player && !mob.Blessed && mob.AccessLevel == AccessLevel.Player && mob.Alive && mob.InLOS(attachedMob))
                decimateHim = true;

            if (decimateHim)
            {
                mob.RevealingAction();
                attachedMob.MovingParticles(mob, 0x379F, 7, 0, false, true, 3043, 4043, 0x211);
                Effects.SendLocationParticles(EffectItem.Create(mob.Location, mob.Map, EffectItem.DefaultDuration), 0x36B0, 1, 14, 63, 7, 9915, 0);
                Effects.PlaySound(mob.Location, mob.Map, 0x229);

                if (!SoutezeLog.IsAbilityImmune(mob))
                {
                    mob.Hits -= dmg;
                    ISTile5.SendDmgPacket(mob, dmg);
                }

                if (mob.Hits < 1)
                    mob.Kill();
                else
                {
                    count++;

                    if (count < 6)
                    {
                        Timer.DelayCall(TimeSpan.FromSeconds(1.3), new TimerStateCallback(DecimateTarget), new object[] { count, mob, attachedMob, dmg + 5 });
                        return;
                    }
                }
            }

            mob.SendMessage(61, "Decimate ended!");
        }
        #endregion

        #region BANISH
        private void TriggerBanish(object state)
        {
            object[] args = (object[])state;
            Mobile mob = (Mobile)args[0];

            if (mob == null || mob.Deleted || !mob.Alive)
                return;

            new FlameWaveTimer(mob).Start();

            mob.CantWalk = false;
            ArrayList targets = Utilitky.GetTargets(mob, 15, false);
            // 			mob.FixedParticles( 0x3709, 10, 30, 5052, EffectLayer.LeftFoot );

            for (int i = 0; i < targets.Count; ++i)
            {
                Mobile m = (Mobile)targets[i];

                if (m == null || m.Deleted)
                    continue;

                Timer.DelayCall(TimeSpan.FromMilliseconds(150 * (GetDist(mob.Location, m.Location) / 3)), new TimerStateCallback(Hurt), new object[] { m, mob });
            }
        }

        public void Hurt(object state)
        {
            object[] args = (object[])state;
            Mobile m = (Mobile)args[0];
            Mobile mob = (Mobile)args[1];

            if (m == null || mob == null || !mob.CanBeHarmful(m) || !m.InLOS(mob))
                return;

            m.RevealingAction();
            mob.DoHarmful(m);

            //FixedParticles( int itemID, int speed, int duration, int effect, int hue, int renderMode, EffectLayer layer )
            m.FixedParticles(0x3709, 1, 30, 9965, 5, 7, EffectLayer.Waist);
            m.FixedParticles(0x376A, 1, 30, 9502, 5, 3, EffectLayer.Waist);
            m.PlaySound(0x231);

            if (!SoutezeLog.IsAbilityImmune(m))
            {
                ISTile5.SendDmgPacket(m, (m.Hits - 1));
                m.Hits = 1;
            }
        }

        public static double GetDist(Point3D start, Point3D end)
        {
            int xdiff = start.X - end.X;
            int ydiff = start.Y - end.Y;
            return Math.Sqrt((xdiff * xdiff) + (ydiff * ydiff));
        }

        internal class FlameWaveTimer : Timer
        {
            private Mobile m_From;
            private Point3D m_StartingLocation;
            private Map m_Map;
            private int m_Count;
            private Point3D m_Point;

            public FlameWaveTimer(Mobile from)
                : base(TimeSpan.FromMilliseconds(150.0), TimeSpan.FromMilliseconds(150.0))
            {
                this.m_From = from;
                this.m_StartingLocation = from.Location;
                this.m_Map = from.Map;
                this.m_Count = 0;
                this.m_Point = new Point3D();
            }

            protected override void OnTick()
            {
                if (this.m_From == null || this.m_From.Deleted || m_Map == null)
                {
                    this.Stop();
                    return;
                }

                double dist = 0.0;
                Point3D fromEye = m_StartingLocation;
                fromEye.Z += 14;

                for (int i = -this.m_Count; i < this.m_Count + 1; i++)
                {
                    for (int j = -this.m_Count; j < this.m_Count + 1; j++)
                    {
                        this.m_Point.X = this.m_StartingLocation.X + i;
                        this.m_Point.Y = this.m_StartingLocation.Y + j;
                        this.m_Point.Z = this.m_Map.GetAverageZ(this.m_Point.X, this.m_Point.Y);
                        dist = GetDist(this.m_StartingLocation, this.m_Point);

                        if ((dist < ((double)this.m_Count + 0.1) && dist > ((double)this.m_Count - 3.1)) && m_Map.LineOfSight(fromEye, m_Point))
                        {
                            //Effects.SendLocationParticles(EffectItem.Create(this.m_Point, this.m_Map, EffectItem.DefaultDuration), 0x3709, 10, 30, 5052);
                            Effects.SendLocationParticles(EffectItem.Create(this.m_Point, this.m_Map, EffectItem.DefaultDuration), 0x3709, 10, 30, 5, 7, 9965, 0);
                        }
                    }
                }

                this.m_Count += 3;

                if (this.m_Count > 15)
                    this.Stop();
            }
        }
        #endregion

        #region POOL OF POISON
        private void TriggerPoisonPool(object state)
        {
            object[] args = (object[])state;
            Mobile mob = (Mobile)args[0];

            if (mob == null || mob.Deleted || !mob.Alive)
                return;

            mob.CantWalk = false;
            int minusrang = (0 - 4);
            Map map = mob.Map;
            Point3D p = mob.Location;
            
            if (map != null)
            {
                for (int x = minusrang; x <= 4; ++x)
                {
                    for (int y = minusrang; y <= 4; ++y)
                    {
                        double dist = Math.Sqrt(x * x + y * y);

                        if (dist <= 4)
                        {
                            p.X = mob.X + x;
                            p.Y = mob.Y + y;

                            IPoint3D o = p as IPoint3D;

                            Spells.SpellHelper.GetSurfaceTop(ref o);

                            p = new Point3D(o);

                            if (!mob.InLOS(p))
                                continue;

                            PoolOfAcid acid = new PoolOfAcid(TimeSpan.FromSeconds(10.0), 250, 300);
                            acid.MoveToWorld(p, map);
                        }
                    }
                }
            }
        }
        #endregion

        #region LIVING BOMB
        private void TriggerLivingBomb(Mobile mob)
        {
            if (mob == null || mob.Deleted || !mob.Alive)
                return;

            ArrayList targets = new ArrayList();

            foreach (Mobile m in mob.GetMobilesInRange(10))
            {
                if (m == null || m.Deleted)
                    continue;

                if (m.Player && !m.Blessed && m.AccessLevel == AccessLevel.Player && m.Alive && m.InLOS(mob))
                    targets.Add(m);
            }

            if (targets.Count < 1)
                return;

            Mobile target = (Mobile)targets[Utility.Random(targets.Count)];

            if (target == null || target.Deleted || !target.Alive)
                return;

            mob.PublicOverheadMessage(MessageType.Regular, m_Hue, true, String.Format("Living Bomb on {0}!", target.RawName));
            target.PublicOverheadMessage(MessageType.Regular, 0x22, false, "Living Bomb on me!");
            AbilitySound(mob);
            target.FixedParticles(0x3728, 1, 13, 0x26B8, 0x455, 7, EffectLayer.Waist);
            target.FixedParticles(0x3779, 1, 15, 0x251E, 0x3F, 7, EffectLayer.Waist);

            target.RevealingAction();
            target.SolidHueOverride = 2019;
            mob.DoHarmful(target);

            Timer.DelayCall(TimeSpan.FromSeconds(2.5), new TimerStateCallback(LivingBombTick), new object[] { 3, target, mob });
        }

        private void LivingBombTick(object state)
        {
            object[] args = (object[])state;
            int count = (int)args[0];
            Mobile mob = (Mobile)args[1];
            Mobile attachedMob = (Mobile)args[2];

            if (attachedMob == null || attachedMob.Deleted || !attachedMob.Alive || mob == null || mob.Deleted || !mob.Alive || !attachedMob.InRange(mob.Location, 25))
            {
                mob.SolidHueOverride = -1;
                return;
            }

            if (count > 0)
            {
                mob.PublicOverheadMessage(MessageType.Regular, 0x22, false, count.ToString());
                count--;
                Timer.DelayCall(TimeSpan.FromSeconds(1.5), new TimerStateCallback(LivingBombTick), new object[] { count, mob, attachedMob });
            }
            else //Detonate
            {
                ArrayList targets = Utilitky.GetTargets(mob, 6, true);

                for (int i = 0; i < targets.Count; ++i)
                {
                    Mobile m = (Mobile)targets[i];

                    if (m == null || m.Deleted || m == mob)
                        continue;

                    m.RevealingAction();
                    attachedMob.DoHarmful(m);

                    int auradamage = PercentualDamage(100, m);
                    int scalebyrange;

                    if (m.InRange(mob.Location, 1))
                        scalebyrange = 1;
                    else if (m.InRange(mob.Location, 4))
                        scalebyrange = 2;
                    else
                        scalebyrange = 3;

                    auradamage /= scalebyrange;

                    m.FixedParticles(0x36BD, 20, 10, 5044, EffectLayer.Head);
                    m.PlaySound(0x307);

                    ISTile5.SendDmgPacket(m, auradamage);
                    m.Hits -= auradamage;

                    if (m.Hits < 1)
                        m.Kill();
                }

                int dmg = PercentualDamage(75, mob);

                if (SoutezeLog.IsAbilityImmune(mob))
                    dmg = 1;

                mob.RevealingAction();
                attachedMob.DoHarmful(mob);
                mob.Hits -= dmg;
                ISTile5.SendDmgPacket(mob, dmg);
                mob.SolidHueOverride = -1;
                mob.FixedParticles(0x36BD, 20, 10, 5044, EffectLayer.Head);
                mob.PlaySound(0x307);
                if (mob.Hits < 1)
                    mob.Kill();
            }
        }
        #endregion

        #region FLARE VORTEX
        private void SpawnFlare(object state)
        {
            object[] args = (object[])state;
            FlareVortex flare = (FlareVortex)args[0];

            if (flare != null)
            {
                flare.MoveToWorld((Point3D)args[1], (Map)args[2]);
            }
        }
        #endregion

        #region MOLTEN GROUND
        private void TriggerMolten(object state)
        {
            object[] args = (object[])state;
            Mobile mob = (Mobile)args[0];

            if (mob == null || mob.Deleted || !mob.Alive)
                return;

            mob.CantWalk = false;
            Point3D p = mob.Location;
            Effects.PlaySound(p, mob.Map, 1140);

            for (int x = p.X - 10; x <= p.X + 10; x++)
            {
                SpawnMoltenGround(p, x, p.Y, mob);
                SpawnMoltenGround(p, x, p.Y + 1, mob);
            }

            for (int y = p.Y - 10; y <= p.Y + 10; y++)
            {
                SpawnMoltenGround(p, p.X, y, mob);
                SpawnMoltenGround(p, p.X - 1, y, mob);
            }
        }

        private void SpawnMoltenGround(Point3D p, int x, int y, Mobile caster)
        {
            if (x == 0 && y == 0)
                return;

            p.X = x;
            p.Y = y;

            IPoint3D o = p as IPoint3D;

            Server.Spells.SpellHelper.GetSurfaceTop(ref o);

            p = new Point3D(o);

            if (!caster.InLOS(p))
                return;

            PoolOfAcid acid = new PoolOfAcid(TimeSpan.FromSeconds(10.0), 100, 150);
            acid.Hue = 2020;
            acid.Name = "molten ground";
            acid.MoveToWorld(p, caster.Map);
        }
        #endregion

        #region COLD BLOOD
        private void TriggerColdBlood(object state)
        {
            object[] args = (object[])state;
            Mobile mob = (Mobile)args[0];

            if (mob == null || mob.Deleted || !mob.Alive)
                return;

            mob.CantWalk = false;
            Server.Ability.ColdBlood(mob);
        }
        #endregion
    }

    //============ ITEMS
    public class FlareVortex : Item
    {
        [Constructable]
        public FlareVortex() : base(14217)
        {
            Movable = false;
            Hue = 2021;
            Name = "flare vortex";

            Timer.DelayCall(TimeSpan.FromSeconds(6.0), new TimerCallback(Morph));
        }

        public override bool OnMoveOver(Mobile m)
        {
            if (!m.Player || !m.Alive || m.Blessed || m.AccessLevel > AccessLevel.Player)
                return true;

            DestroyIt(m);
            return true;
        }

        private void DestroyIt(Mobile m)
        {
            if (this.Deleted)
                return;

            int dmg = XmlSpecial.PercentualDamage(20, m);
            m.RevealingAction();

            if (SoutezeLog.IsAbilityImmune(m))
                dmg = 1;

            m.Hits -= dmg;
            ISTile5.SendDmgPacket(m, dmg);
            Effects.SendLocationParticles(EffectItem.Create(Location, Map, EffectItem.DefaultDuration), 14000, 9, 32, 5024);
            Effects.PlaySound(Location, Map, 285);

            if (m.Hits < 1)
                m.Kill();

            this.Delete();
        }

        public override void OnDoubleClick(Mobile from)
        {
            if (!from.Player || from.AccessLevel > AccessLevel.Player || from.Hidden || !from.Alive || from.Blessed)
                return;

            if (!from.InRange(this.GetWorldLocation(), 1 + Server.Regions.KhaldunRegion.Talisman(from, 2)))
                from.SendLocalizedMessage(502138);
            else
                DestroyIt(from);
        }

        private void Morph()
        {
            if (this.Deleted)
                return;

            if (ItemID == 6006)
            {
                ItemID = 6008;
                Hue = 2487;
            }
            else
            {
                ItemID = 14284;
                Hue = 2019;
            }

            Effects.PlaySound(Location, Map, 264);

            Timer.DelayCall(TimeSpan.FromSeconds(3.0), new TimerStateCallback(Explode), new object[] { 3 });
        }

        private void Explode(object state)
        {
            if (this.Deleted)
                return;

            object[] args = (object[])state;
            int count = (int)args[0];

            if (count > 0)
            {
                this.PublicOverheadMessage(MessageType.Regular, 0x22, false, count.ToString());
                count--;
                Timer.DelayCall(TimeSpan.FromSeconds(2.0), new TimerStateCallback(Explode), new object[] { count });
            }
            else //Detonate
            {
                ArrayList targets = new ArrayList();

                foreach (Mobile m in this.GetMobilesInRange(10))
                {
                    if (m == null || m.Deleted)
                        continue;

                    if (m is BaseCreature && (((BaseCreature)m).Controlled || ((BaseCreature)m).Summoned))
                    {
                        if (!m.IsDeadBondedPet && m.Alive && !m.Blessed && m.InLOS(this))
                            targets.Add(m);
                    }
                    else if (m.Player && !m.Blessed && m.AccessLevel == AccessLevel.Player && m.Alive && m.InLOS(this))
                        targets.Add(m);
                }

                for (int i = 0; i < targets.Count; ++i)
                {
                    Mobile m = (Mobile)targets[i];

                    if (m == null || m.Deleted)
                        continue;

                    m.RevealingAction();
                    m.FixedParticles(0x36BD, 20, 10, 5044, EffectLayer.Head);
                    m.PlaySound(284);
                    Effects.SendMovingEffect(this, m, 0x36D4, 5, 0, false, false);

                    if (m is BaseCreature)
                        m.Kill();
                    else if (SoutezeLog.IsAbilityImmune(m))
                    {
                    }
                    else
                    {
                        ISTile5.SendDmgPacket(m, (m.Hits - 1));
                        m.Hits = 1;
                    }
                }

                Effects.SendLocationParticles(EffectItem.Create(Location, Map, EffectItem.DefaultDuration), 14000, 9, 32, 5024);
                Effects.PlaySound(Location, Map, 285);
                this.Delete();
            }
        }

        public FlareVortex(Serial serial) : base(serial)
        {
        }

        public override void Serialize(GenericWriter writer)
        {
            base.Serialize(writer);
            writer.Write((int)0); // version
        }

        public override void Deserialize(GenericReader reader)
        {
            base.Deserialize(reader);
            int version = reader.ReadInt();

            this.Delete();
        }
    }
}
